/*
 * esl.c - Lava Etherserial Link driver for kernel 2.6.x
 * This driver virtualize remote serial port to represent it as local 
 * regular serial port.
 *
 * 
 * Copyright (C) Lava Computer Mfg. Inc. All rights reserved.
 * Author: Eduard G. Kromskoy    <ed1k@lavalink.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Modified and updated for use with kernel version 2.6.22.4-65.fc7 (fedora)
 * By Richard Burgan September 22, 2007
 *
 * Modified to install in older style from Richards fix.
 * By Ian MacPherson - LAVA Computer MFG. Nov 6, 2007
 * 
 */


#include <linux/module.h>
//#include <linux/config.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
//#include <linux/moduleparam.h>
#include <linux/param.h>
#include <linux/signal.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/smp_lock.h>
#include <linux/net.h>
#include <linux/circ_buf.h>
#include <net/sock.h>
#include <asm/byteorder.h>
#include <asm/atomic.h>
#include <asm/page.h>
#include <asm/processor.h>
#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <linux/termios.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/tty_driver.h>
#include <linux/slab.h>

#include <linux/in.h>

#include <linux/serial.h>    // for serial_icounter_struct

typedef    uint16_t    in_port_t;
typedef    uint32_t    in_addr_t;

#define NR_PORTS        64
#define LAVAVERSION     "0.22"
#define LAVADATE        "Nov 6, 2007"
#define LAVADRIVERNAME  "Lava Etherserial Link driver"
#define LAVAMAJOR       247

//if you uncomment it your /var/log/messages will be flooded,
//use with care if you're really sure it helps you to debug a problem.
//#define MDEBUG        1

static int verbose=1;
static int timeout=10;
static int ttymajor=LAVAMAJOR;
static int nr_ports = NR_PORTS; // needed to satisfy module_param_array()'s requirement for a pointer
//default IP address of remote serial port
static char *ip[NR_PORTS]={"192.168.0.35"};
//default TCP port of remote serial port
#define INITPORT        4098
static char *name="ttyES";

/* Variables for insmod */
MODULE_AUTHOR("Eduard Kromskoy, Lava Computer Mfg. Inc.");
MODULE_DESCRIPTION(LAVADRIVERNAME);
MODULE_LICENSE("GPL");

MODULE_ALIAS("char-major-"__MODULE_STRING(LAVAMAJOR));

/* Module Parameters */
#if defined(module_param)			// Showing parameters undef SYSFS (New Style)
module_param(ttymajor, int, 0);
module_param(verbose, int, 0);
module_param_array(ip, charp, &nr_ports, 0);
module_param(timeout, int, 0);
module_param(name, charp, 0);
MODULE_PARM_DESC(ip,"IP:TCPPort[,IP1:TCPPort1][,...] (default ip=192.168.0.80:"__MODULE_STRING(INITPORT)")");
MODULE_PARM_DESC(timeout,"Timeout for waiting response from remote device in HZ/1000 (defaul 10, i.e. HZ/100)");
MODULE_PARM_DESC(name,"Device name (default name=ttyES, so first device is /dev/ttyES0)");
#else                                           // Old Style
MODULE_PARM(ttymajor, "i");
MODULE_PARM(verbose, "i");
MODULE_PARM(ip,"1-"__MODULE_STRING(NR_PORTS)"s");
MODULE_PARM(timeout,"i");
MODULE_PARM(name,"s");
MODULE_PARM_DESC(ip,"IP:TCPPort[,IP1:TCPPort1][,...] (default ip=192.168.0.35:"__MODULE_STRING(INITPORT)")");
MODULE_PARM_DESC(timeout,"Timeout for waiting response from remote device in HZ/1000 (defaul 10, i.e. HZ/100)");
MODULE_PARM_DESC(name,"Device name (default name=ttyES, so first device is /dev/ttyES0)");
#endif

#define PORTNO(x) ((x)->index)
#define RELEVANT_IFLAG(iflag) (iflag&(IGNBRK|BRKINT|IGNPAR|PARMRK|INPCK))

//Lava Etherserial device structure
typedef struct etherserial{
    in_addr_t                ip;        //device ip
    in_port_t                port;      //TCP port
    int                      devno;     //device number
    struct socket            *td;       //socket ptr
    ssize_t                  dsize;     //max data chunk size for write request
    uid_t                    uid;       //owner to rstrict access for others
    uint8_t                  *tcpbuf;   //buffer for TCP communications
    uint8_t                  *usrbuf;   //buffer for user data to send
    struct serial_icounter_struct icount;
    wait_queue_head_t        wait;
    wait_queue_head_t        open_wait;
    wait_queue_head_t        close_wait;
    wait_queue_head_t        delta_msr_wait;
    struct tty_struct        *tty;
    struct circ_buf          xmit;     //tty_struct doesn't have xmit buffer
    spinlock_t               lock;     //lock for safe access to the device
    struct semaphore         mux;      //mutex for accessing the user buffer
    uint16_t                 status;   //device status flags
    uint32_t                 cmd;      //command flags (from TTY)
    } ETHERSERIAL;
 
//the tty driver will allocate memory upon open() request 
//and fill up these arrays of pointers.
static struct tty_struct *lava_table[NR_PORTS];
static struct ktermios *lava_termios[NR_PORTS];
static struct ktermios *lava_termios_locked[NR_PORTS];
//static int lava_refcount;
//driver structure
static struct tty_driver lava_driver;
static char *driver_name="esl";

//array of pointers to lava devices
static ETHERSERIAL *lava_dev[NR_PORTS];

//exit maintenance for threads
static atomic_t lava_shutdown_cmd;
static atomic_t LTPCOUNT;
static DECLARE_WAIT_QUEUE_HEAD(lava_wait_threads);

//flow control constants
#define SERIAL_E16_ERROR_ABORT          0x8000
#define SERIAL_E16_DTR_CONTROL          0x0001
#define SERIAL_E16_XOFF_CONTINUE        0x8000
#define SERIAL_E16_RTS_CONTROL          0x0040
#define SERIAL_E16_CTS_HANDSHAKE        0x0008
#define SERIAL_E16_RTS_HANDSHAKE        0x0080
#define SERIAL_E16_AUTO_TRANSMIT        0x0001
#define SERIAL_E16_AUTO_RECEIVE         0x0002

//events
#define SERIAL_EV_CTS                    0x0008
#define SERIAL_EV_DSR                    0x0010
#define SERIAL_EV_RLSD                   0x0020    //DCD
#define SERIAL_EV_RING                   0x0100

//error messages for cSRErrorComPortNMessage (0x1203)
#define SERIAL_E16_LSR_BI                1
#define SERIAL_E16_LSR_FE                2
#define SERIAL_E16_LSR_OVR               4
#define SERIAL_E16_LSR_RxOVR             8
#define SERIAL_E16_LSR_PE                16

//
//commands which we can get from tty layer
//
#define LAVA_CMD_SETBRK                (1<<1)
#define LAVA_CMD_CLRBRK                (1<<2)
#define LAVA_CMD_SETDTR                (1<<3)
#define LAVA_CMD_CLRDTR                (1<<4)
#define LAVA_CMD_SETRTS                (1<<5)
#define LAVA_CMD_CLRRTS                (1<<6)
#define LAVA_CMD_CLOSE                 (1<<15)
#define LAVA_CMD_OPEN                  (1<<16)
#define LAVA_CMD_INIT                  (1)
#define LAVA_CMD_RESUME                (1<<17)
#define LAVA_CMD_SUSPEND               (1<<7)

//commands which we're doing internally in Lava transport protocol handler
//implementing steps for stty command
#define LAVA_CMD_SETCC                 (1<<8)        //set control chars
#define LAVA_CMD_SETFC                 (1<<9)        //set flow control
#define LAVA_CMD_SETBAUD               (1<<10)
#define LAVA_CMD_SETLCR                (1<<11)
#define LAVA_CMD_SETMCR                (1<<12)

//errors & events recovery
#define LAVA_CMD_ERROR                 (1<<13)
#define LAVA_CMD_EVENT                 (1<<14)

//
//status flags 
//
#define LAVA_STS_DRAINED            0x0080
#define LAVA_STS_OPEN               0x0040
#define LAVA_STS_DTR                0x0001
#define LAVA_STS_RTS                0x0002
#define LAVA_STS_DCD                0x8000
#define LAVA_STS_RI                 0x4000
#define LAVA_STS_DSR                0x2000
#define LAVA_STS_CTS                0x1000


//
// List of requests which are common for LPP and Remote System
// (lpp_ser.h in windows project)

#define cRSGetAccessRequest                     0x0400
#define cRSReleaseAccessRequest                 0x0401

//
// List of Request/Messages which notify system about status
//      of transport
//

#define cRSTransportOffRequest                   0x0402 
#define cRSTransportOnMessage                    0x0403

//
// List of request which are specific for Serial Service
//

//
// List of Critical Request/Respond
//
#define cRSOpenComPortCRequest                   0x4200
#define cRSCloseComPortCRequest                  0x4201

#define cRSResetComPortCRequest                  0x4202

#define cRSGetBaudRateComPortCRequest            0x4203
#define cRSGetLineControlComPortCRequest         0x4204
#define cRSGetModemControlComPortCRequest        0x4205
#define cRSGetLineStatusComPortCRequest          0x4206
#define cRSGetModemStatusComPortCRequest         0x4207

#define cRSSetBaudRateComPortCRequest            0x4208
#define cRSSetLineControlComPortCRequest         0x4209
#define cRSSetModemControlComPortCRequest        0x420A

#define cRSSetRtsComPortCRequest                 0x420B
#define cRSClrRtsComPortCRequest                 0x420C
#define cRSSetDtrComPortCRequest                 0x420D
#define cRSClrDtrComPortCRequest                 0x420E

#define cRSSetEscapeCharComPortCRequest          0x420F
#define cRSSetSpecialCharsComPortCRequest        0x4210
#define cRSSetHandFlowComPortCRequest            0x4211

#define cRSWriteImmediateComPortCRequest         0x4212
#define cRSSendXoffComPortCRequest               0x4213
#define cRSSendXonComPortCRequest                0x4214
#define cRSTurnOffBreakComPortCRequest           0x4215
#define cRSTurnOnBreakComPortCRequest            0x4216

#define cRSFlushComPortCRequest                  0x4217
#define cRSEnableCaptureEventComPortCRequest     0x4218

#define cRSClrStatisticsComPortCRequest          0x4219
#define cRSGetStatisticsComPortCRequest          0x421A

#define cRSGetCommStatusComPortCRequest          0x421B

#define cRSRequiredCompleteReadComPortCRequest   0x421C

#define cRSPurgeRxComPortCRequest                0x421D
#define cRSPurgeTxComPortCRequest                0x421E
#define cRSPurgeTxRxComPortCRequest              0x421F

//
// List of Non Critical Request/Messages
//

#define cRSWriteComPortNRequest                  0x1100
#define cRSCompleteWriteComPortNRequest          0x1101
#define cRSReceiveDataComPortNRequest            0x1102
#define cRSRequiredReadAmountComPortNRequest     0x1103

#define cRSSuspendShipRXComPortNRequest          0x1200
#define cRSResumeShipRXComPortNRequest           0x1201

#define cRSEventAcceptedComPortNRequest          0x1202

#define cRSErrorComPortNMessage                  0x1203
#define cRSClearErrorComPortNRequest             0x1204

#define cRSEventComPortNMessage                  0x1205
#define cRSSetMaskEventComPortNRequest           0x1206
#define cRSSetHistoryEventMaskComPortNRequest    0x1207
#define cRSEnableCaptureEventComPortNRequest     0x1208
#define cRSClrHistoryEventMaskComPortNRequest    0x1209

//
// List of Fixed Synchronous Service Request/Response
//

#define cRSSynchTerminatorLPP                    0x0000
//--end of lpp_ser.h

// M$ sais it helps, let it be here
#define BEGIN_MESSAGE_MAP()                     if(0){
#define ON_COMMAND(cond)                         }else if(cond){ 
#define END_MESSAGE_MAP()                        }


//quantity of empty packets (syncs) before shutting transport off 
#define MAXEMPTYPACKS                    10
//gives timeout for periodicaly connection check when 
//there is no data exchange in place
#define WATCHDOG                         420
//quantity of "response timeout" till we decide to reconnect
//it depends on timeout and distance to remote box.
//for boxes placed far away over internet it is
//better to use parameter timeout=20 or even more 
//when inserting module.
#define MAXTIMEOUTS                      250

//wake up someone who wants write to the device
//if there are less than WAKEUP_CHARS in xmit buffer
#define WAKEUP_CHARS                     256

static void lava_flush_buffer(struct tty_struct * tty);
static void lava_wait_until_sent(struct tty_struct *tty, int timeout);
static int lava_open(struct tty_struct *tty, struct file * filp);
static void lava_close(struct tty_struct *tty, struct file * filp);
static void lava_throttle(struct tty_struct *tty);
static void lava_unthrottle(struct tty_struct *tty);
static int lava_write(struct tty_struct *tty, const unsigned char * buf, int count);
static int lava_write_room(struct tty_struct * tty);
static int lava_chars_in_buffer(struct tty_struct * tty);
static void lava_flush_chars(struct tty_struct * tty);
static int lava_ioctl(struct tty_struct *tty, struct file * file,
            unsigned int cmd, unsigned long arg);
static void lava_set_termios(struct tty_struct *tty, struct ktermios *old_termios);
static void lava_stop(struct tty_struct *tty);
static void lava_start(struct tty_struct *tty);
static void lava_hangup(struct tty_struct *tty);
static void lava_break(struct tty_struct *tty, int break_state);
static void lava_send_xchar(struct tty_struct *tty, char ch);
static int lava_read_proc(char *page, char **start, off_t off, int count,
            int *eof, void *data);

static struct tty_operations lava_ops={
    .open=lava_open,
    .close=lava_close,
    .write=lava_write,
    .write_room=lava_write_room,
    .flush_buffer=lava_flush_buffer,
    .chars_in_buffer=lava_chars_in_buffer,
    .flush_chars=lava_flush_chars,
    .ioctl=lava_ioctl,
    .throttle=lava_throttle,
    .unthrottle=lava_unthrottle,
    .set_termios=lava_set_termios,
    .stop=lava_stop,
    .start=lava_start,
    .hangup=lava_hangup,
    .read_proc=lava_read_proc,
    .break_ctl=lava_break,
    .wait_until_sent=lava_wait_until_sent,
    .send_xchar=lava_send_xchar,
};

/* ************************************************************************** */
static inline in_addr_t 
ke_inet_addr(char *addrstr){
//returns IP(v4) in network byte order from IP string
    union {
        uint8_t byte[4];
        uint32_t dword;
    } ip;
    char *optnext=addrstr;
    int i;    
    
    ip.dword=0;
    for(i=0;i<4;i++){
        ip.byte[i]=(uint8_t)simple_strtoul(optnext,&optnext,0);
        if(!optnext) break;
        if(*optnext=='.') optnext++;
        else break;
    }
    return (in_addr_t)((i==3)?ip.dword:0);
}

/* ************************************************************************** */
static inline int 
ke_inet_name(char *addrstr, in_addr_t ipaddr){
//converts numeric IP(v4) into string representation
//resulting string is stored in callers buffer addrstr[]
//returns length of string
    union {
        uint8_t byte[4];
        uint32_t dword;
    } ip;
    int i, ret=0;
    ip.dword=(uint32_t)ipaddr;
    for(i=0;i<4;i++)
        ret+=sprintf(addrstr+ret, "%d.", ip.byte[i]);
    ret--;
    *(addrstr+ret)=0;
    return ret;
}

/* ************************************************************************** */
static struct socket* 
OpenTunnel(in_addr_t ip, in_port_t TCPport){
//returns pointer to the socket for tunnel (TCP connection)
    struct socket *sock = NULL;
    struct sockaddr_in sa;
    
    if(sock_create(PF_INET, SOCK_STREAM, IPPROTO_TCP, &sock)<0){
    //can't create TCP socket --return
#ifdef MDEBUG
printk(KERN_INFO"etherserial->OpenTunnel: Can not create TCP socket.\n");
#endif
        return NULL;
    }
    memset(&sa, 0, sizeof(sa));
    sa.sin_family=AF_INET;
    sa.sin_port=htons((unsigned short)TCPport);
    sa.sin_addr.s_addr=ip;
#ifdef MDEBUG        
printk(KERN_INFO"etherserial->OpenTunnel: %x:%d[TCP]\n", be32_to_cpu(sa.sin_addr.s_addr),be16_to_cpu(sa.sin_port));
#endif
    if(sock->ops->connect(sock, (struct sockaddr*) &sa, sizeof(sa), 0)!=0){
        if(verbose>1){
            printk(KERN_INFO"etherserial->OpenTunnel: can not connect to %x:%d.\n",
            be32_to_cpu(sa.sin_addr.s_addr),be16_to_cpu(sa.sin_port));
        }
        sock_release(sock);
        return NULL;
    }
    return sock;
}

/* ************************************************************************** */
static void 
CloseTunnel(struct socket *sock){
//might be need to read everything from socket as in tHTTPd
    sock_release(sock);
}

/* ************************************************************************** */
//Lava Transport Protocol Daemon (kernel thread).
//Init routine will start a daemon per remote serial port.
static int 
ltp(void *ptr){

    ETHERSERIAL           *dev=(ETHERSERIAL*)ptr;
    struct task_struct    *me=current;
    uint16_t              *inpacket;
    uint16_t              *outpacket;
    int                   inpacketsize, outpacketsize;
    int                   TransportOffFlag=0;
    volatile uint32_t     buzy=0;
    int                   emptytransaction=MAXEMPTYPACKS;
    int                   timeouts=MAXTIMEOUTS;
    int                   watchdogtimer=WATCHDOG;
    int                   txdisable=0;
    uint32_t              i, status;
    uint16_t              msgid, len, msr, oldmsr;
    uint8_t               *bptr;
    uint32_t              irp=0x12345678;    //windows rudiment
    int                   n;
    struct msghdr         msg;
    struct iovec          iov;
    mm_segment_t          oldfs;
        
    inpacket  = (uint16_t*)dev->tcpbuf;
    outpacket = (uint16_t*)dev->tcpbuf;    
    memset(&msg, 0, sizeof(msg));
    memset(&iov, 0, sizeof(iov));
    
    //close fds and disconnect from console
    daemonize("ltpd-%02d", dev->devno);
    
    spin_lock_irq(&me->sighand->siglock);
    siginitsetinv(&me->blocked, sigmask(SIGKILL)|sigmask(SIGTERM));
    recalc_sigpending();
    spin_unlock_irq(&me->sighand->siglock);

    //main loop of lava transport protocol thread    
    while(!atomic_read(&lava_shutdown_cmd)&&!signal_pending(me)){
start:    
        spin_lock(&dev->lock);
        status=dev->cmd|(LAVA_STS_OPEN&dev->status);
        spin_unlock(&dev->lock);    
        status|=buzy;
        if(!status) 
            interruptible_sleep_on(&dev->wait);
        else
            interruptible_sleep_on_timeout(&dev->wait, HZ*timeout/1000);

        if(!dev->td){
            //no TCP connection
            dev->td=OpenTunnel(dev->ip, dev->port);
            if(!dev->td){
                //fail to connect
                continue;
            } else if(dev->td->sk) dev->td->sk->sk_reuse=1;
            //reopen the port if it was opened
            if(status&LAVA_STS_OPEN) status=buzy=LAVA_CMD_OPEN;
            txdisable=0;
            emptytransaction=MAXEMPTYPACKS;
            timeouts=MAXTIMEOUTS;
            watchdogtimer=WATCHDOG;            
        }
        
        //we are connected
        //recieve incoming message
        
        msg.msg_flags         = MSG_DONTWAIT|MSG_NOSIGNAL;
        msg.msg_name          = NULL;
        msg.msg_namelen       = 0;
        msg.msg_control       = NULL;
        msg.msg_controllen    = 0;
        msg.msg_iov           = &iov;
        msg.msg_iovlen        = 1;
        msg.msg_iov->iov_base = (char*)inpacket;
        msg.msg_iov->iov_len  = (__kernel_size_t)PAGE_SIZE;
        oldfs = get_fs(); set_fs(KERNEL_DS);
        inpacketsize = sock_recvmsg(dev->td,&msg,PAGE_SIZE,MSG_DONTWAIT);
        set_fs(oldfs);

        if(inpacketsize != -EAGAIN){
            //get packet        
            if(inpacketsize < 4){
                //we got bogus message or connection dropped
                CloseTunnel(dev->td);
                dev->td=NULL;
                continue;
            } 
            bptr=(uint8_t*)inpacket;
            timeouts=MAXTIMEOUTS;
#ifdef MDEBUG    
if(inpacket[0]){ //filter out syncs
printk(KERN_INFO"%s: RETURNED MESSAGE (hex) InPacket:", me->comm);
for (i=0; i<inpacketsize; i++) printk(" %02x", *((uint8_t*)inpacket + i));
printk("\n");
}
#endif
            if(inpacket[0] != 0){
                //got non-empty packet 
                emptytransaction=MAXEMPTYPACKS;
                TransportOffFlag=0;
            } else if(emptytransaction) emptytransaction--;
                
/*
 *  Loop through message
 */
            while((bptr-(uint8_t*)inpacket) < inpacketsize){
                
                msgid=*((uint16_t*)bptr);
                bptr+=2;
                len=*((uint16_t*)bptr);
                bptr+=2;
                msgid=le16_to_cpu(msgid);
                len=le16_to_cpu(len);
                
                switch(msgid){

                case cRSOpenComPortCRequest:           //open comm port response
                    dev->dsize=le16_to_cpu(*((uint16_t*)bptr));
                    if(!dev->dsize){
                        //error
                        CloseTunnel(dev->td);
                        dev->td=NULL;
                        goto start;                    //reconnect
                    }
                    bptr+=len;
                    buzy|=LAVA_CMD_INIT;
                break;
                
                case cRSTransportOffRequest:           //transport suspended
                    TransportOffFlag=1;
                    bptr+=len;
                break;
            
                case cRSClrDtrComPortCRequest:         //clear DTR done
                    spin_lock(&dev->lock);
                    dev->status&=~LAVA_STS_DTR;
                    spin_unlock(&dev->lock);
                    buzy&=~LAVA_CMD_CLRDTR;                    
                    bptr+=len;
                break;
                    
                case cRSClrRtsComPortCRequest:         //clear RTS done
                    spin_lock(&dev->lock);
                    dev->status&=~LAVA_STS_RTS;
                    spin_unlock(&dev->lock);
                    buzy&=~LAVA_CMD_CLRRTS;                    
                    bptr+=len;
                break;
                    
                case cRSTurnOnBreakComPortCRequest:    //turn on Break done
                    buzy&=~LAVA_CMD_SETBRK;
                    bptr+=len;
                break;
                    
                case cRSTurnOffBreakComPortCRequest:  //turn off Break done
                    buzy&=~LAVA_CMD_CLRBRK;
                    bptr+=len;
                break;
                    
                case cRSSetDtrComPortCRequest:        //set DTR done
                    spin_lock(&dev->lock);
                    dev->status|=LAVA_STS_DTR;
                    spin_unlock(&dev->lock);
                    buzy&=~LAVA_CMD_SETDTR;
                    bptr+=len;
                break;
                    
                case cRSSetRtsComPortCRequest:        //set RTS done
                    spin_lock(&dev->lock);
                    dev->status|=LAVA_STS_RTS;
                    spin_unlock(&dev->lock);
                    buzy&=~LAVA_CMD_SETRTS;
                    bptr+=len;
                break;
                    
                case cRSResetComPortCRequest:
                    buzy&=~LAVA_CMD_INIT;
                    buzy|=LAVA_CMD_SETCC;
                    bptr+=len;
                break;
                
                case cRSSetSpecialCharsComPortCRequest:    
                    //doing STTY, control chars setted up, set flow control is next step
                    buzy&=~LAVA_CMD_SETCC;
                    buzy|=LAVA_CMD_SETFC;
                    bptr+=len;    
                break;
                    
                case cRSSetHandFlowComPortCRequest:
                //doing STTY, flow control setted up, set baud is next step
                    buzy&=~LAVA_CMD_SETFC;
                    buzy|=LAVA_CMD_SETBAUD;
                    bptr+=len;
                break;
                
                case cRSSetBaudRateComPortCRequest:
                //doing STTY, baud rate setted, set LCR is next step
                    buzy&=~LAVA_CMD_SETBAUD;
                    buzy|=LAVA_CMD_SETLCR;
                    bptr+=len;
                  break;
                
                case cRSSetLineControlComPortCRequest:
                //doing STTY, LCR setted, set MCR is next step
                    buzy&=~LAVA_CMD_SETLCR;
                    buzy|=LAVA_CMD_SETMCR;
                    bptr+=len;
                break;
                
                case cRSSetModemControlComPortCRequest:
                //STTY done
                    spin_lock(&dev->lock);
                    if(dev->tty->termios->c_cflag&CBAUD)
                        dev->status|=LAVA_STS_DTR|LAVA_STS_RTS;
                    else 
                        dev->status&=~(LAVA_STS_DTR|LAVA_STS_RTS);
                    spin_unlock(&dev->lock);
                    buzy&=~LAVA_CMD_SETMCR;
                    bptr+=len;
                    if(buzy&LAVA_CMD_OPEN){
                    //it was stty on open, so now we can tell tty the port is open 
                    //and it can do whatever it wants
                        buzy&=~LAVA_CMD_OPEN;
                        spin_lock(&dev->lock);
                        dev->status|=LAVA_STS_OPEN|LAVA_STS_DRAINED;
                        spin_unlock(&dev->lock);
                        wake_up(&dev->open_wait);
                        wake_up_interruptible(&dev->tty->write_wait);
                    }
                    buzy|=LAVA_CMD_EVENT;
                break;
               
//RDB MODIFICATION
//flip buffer
                case cRSReceiveDataComPortNRequest:    //received data
                    while( len > 0 ){                  //add to flip buffer 
                      tty_insert_flip_char(dev->tty, *bptr, 0);
                      len--;
                      bptr++;
                    }                                  //end of while
                    tty_flip_buffer_push( dev->tty );  //push flip buffer to driver
                break;
                
                case cRSErrorComPortNMessage:          //line status error
/*
 * The nature of our Etherserial Link device (at the current moment, at least)
 * doesn't allow us to tell tty which character was actually hit by error. In QNX
 * version I put the event into queue for TTY layer whatever it may decide to inform
 * application about some error (sometimes it's better to know that there was some 
 * error even without knowledge which exactly character was affected). 
 * For Linux version I'm only going to update the statistic meawhile.
 */
                     bptr++;
                     spin_lock(&dev->lock);
                     if(*bptr&SERIAL_E16_LSR_BI) dev->icount.brk++;
                    else if(
                       (*bptr&SERIAL_E16_LSR_OVR) ||
                       (*bptr&SERIAL_E16_LSR_RxOVR)
                            ) dev->icount.overrun++;
                    else if(*bptr&SERIAL_E16_LSR_PE) dev->icount.parity++;
                    else if(*bptr&SERIAL_E16_LSR_FE) dev->icount.frame++;
                    spin_unlock(&dev->lock);
                    bptr++;
                    buzy|=LAVA_CMD_ERROR;
                break;
            
                 case cRSEventComPortNMessage:              //modem status changed
                    bptr+=len;
                    buzy|=LAVA_CMD_EVENT;
                break;
                
                case cRSGetModemStatusComPortCRequest:
                    msr=(uint16_t)*bptr;
                    msr*=256;
                    msr&=0xF000;
                    spin_lock(&dev->lock);
                    oldmsr=dev->status;
                    dev->status=msr|(oldmsr&0x00FF);
                    spin_unlock(&dev->lock);
                    oldmsr=msr^(oldmsr&0xF000);
                    if(oldmsr){
                        //there is change in MSR
                        if(oldmsr&LAVA_STS_CTS) dev->icount.cts++;
                        if(oldmsr&LAVA_STS_DSR) dev->icount.dsr++;
                        if(oldmsr&LAVA_STS_DCD) dev->icount.dcd++;
                        if(oldmsr&LAVA_STS_RI) dev->icount.rng++;
                        wake_up_interruptible(&dev->delta_msr_wait);
                        if((oldmsr&LAVA_STS_CTS)&&(dev->tty->termios->c_cflag&CRTSCTS)){
                            //CTS changed
                            if(msr&LAVA_STS_CTS){
                                dev->tty->hw_stopped=0;
                                if(
                                (dev->tty->flags&(1<<TTY_DO_WRITE_WAKEUP))
                                && dev->tty->ldisc.write_wakeup
                                   )(dev->tty->ldisc.write_wakeup)(dev->tty);
                                wake_up_interruptible(&dev->tty->write_wait);
                            } else dev->tty->hw_stopped=1;
                        }
                        if((oldmsr&LAVA_STS_DCD)&&(!(dev->tty->termios->c_cflag&CLOCAL))){
                            //DCD changed
                            if(!(msr&LAVA_STS_DCD)) tty_hangup(dev->tty);
                        }
                    }
                    buzy&=~LAVA_CMD_EVENT;
                    bptr+=len;
                break;
            
                case cRSCompleteWriteComPortNRequest:
                    bptr+=len;
                    txdisable=0;
                    irp++;
                    //TODO: *********************************
                    //move this to "readlcr" response handler
                    //to be sure TSHR is empty
                    spin_lock(&dev->lock);
                    dev->status|=LAVA_STS_DRAINED;
                    spin_unlock(&dev->lock);
                  break;
            
                case cRSWriteComPortNRequest:
                    bptr+=len;
                    txdisable=0;
                break;
                
                case cRSCloseComPortCRequest:
                    buzy=0;
                    spin_lock(&dev->lock);
                    dev->status=0;
                    spin_unlock(&dev->lock);
                    if(dev->td) CloseTunnel(dev->td);
                    dev->td=NULL;
                    wake_up(&dev->close_wait);
                    goto start;
    
                default:                                  //skip uknown command or sync
                    bptr+=len;
                break;        
                }    //end of switch
            }    //end of while
        //check flag for restart instead of goto
        } else{
            //timeout, it's OK when Transport is off,
            //but is not OK when Transport is on
            if(!TransportOffFlag){
#ifdef MDEBUG    
printk(KERN_INFO"%s: Response timeout.\n", me->comm);
#endif
                if(timeouts--<=0){
                    CloseTunnel(dev->td);
                    dev->td=NULL;
                    continue;    //reconnect
                }
                if(!(status&LAVA_CMD_OPEN)) continue;
            }
        }
/*
 * End of incoming message processing
 ***********************************************************************************/

/*
 * If we are not busy we can take commands from TTY layer
 */

        if(!buzy){
            spin_lock(&dev->lock);
            buzy=dev->cmd;
            dev->cmd=0;
            spin_unlock(&dev->lock);
        }

/*
 * Analysis of buzy flags
 *
 */        
        outpacketsize=0;
    
BEGIN_MESSAGE_MAP();

    ON_COMMAND(buzy&LAVA_CMD_CLRDTR);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSClrDtrComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        
    ON_COMMAND(buzy&LAVA_CMD_CLRRTS);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSClrRtsComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
            
    ON_COMMAND(buzy&LAVA_CMD_INIT);
        //send init message
        outpacketsize=18;    //18 bytes total
           memset(outpacket, 0, outpacketsize);
        //set mask for capturing events
           outpacket[0]=cRSSetMaskEventComPortNRequest;
           outpacket[0]=cpu_to_le16(outpacket[0]);
           outpacket[1]=2;    //two bytes mask
           outpacket[1]=cpu_to_le16(outpacket[1]);
           outpacket[2]=SERIAL_EV_CTS|SERIAL_EV_DSR|SERIAL_EV_RLSD|SERIAL_EV_RING;
           outpacket[2]=cpu_to_le16(outpacket[2]);
        //enable capturing mask
           outpacket[3]=cRSEnableCaptureEventComPortNRequest;
           outpacket[3]=cpu_to_le16(outpacket[3]);
        outpacket[5]=cRSResetComPortCRequest;
        outpacket[5]=cpu_to_le16(outpacket[5]);

    ON_COMMAND(buzy&LAVA_CMD_SETCC);
        outpacketsize=14;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSSetSpecialCharsComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        outpacket[1]=6;
        outpacket[1]=cpu_to_le16(outpacket[1]);
        outpacket[2]=EOF_CHAR(dev->tty)|(KILL_CHAR(dev->tty)<<8);
        //No DC3&DC4
        outpacket[4]=START_CHAR(dev->tty)|(STOP_CHAR(dev->tty)<<8);
        
    ON_COMMAND(buzy&LAVA_CMD_SETFC);
        outpacketsize=12;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSSetHandFlowComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        outpacket[1]=4;
        outpacket[1]=cpu_to_le16(outpacket[1]);
        outpacket[2]=SERIAL_E16_ERROR_ABORT|SERIAL_E16_DTR_CONTROL;
        outpacket[3]=SERIAL_E16_XOFF_CONTINUE|SERIAL_E16_RTS_CONTROL;
        if(dev->tty->termios->c_cflag&CRTSCTS){
            //hw flow control
            outpacket[2]|=SERIAL_E16_CTS_HANDSHAKE;
            outpacket[3]|=SERIAL_E16_RTS_HANDSHAKE;
            outpacket[3]&=~SERIAL_E16_RTS_CONTROL;
        }
        if(dev->tty->termios->c_iflag&IXOFF){
            //sw flow control
            outpacket[3]|=SERIAL_E16_AUTO_TRANSMIT|SERIAL_E16_AUTO_RECEIVE;
            outpacket[3]&=~SERIAL_E16_RTS_CONTROL;
        }
        outpacket[2]=cpu_to_le16(outpacket[2]);
        outpacket[3]=cpu_to_le16(outpacket[3]);
        
    ON_COMMAND(buzy&LAVA_CMD_SETBAUD);
        outpacketsize=12;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSSetBaudRateComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        outpacket[1]=4;
        outpacket[1]=cpu_to_le16(outpacket[1]);
        i=tty_get_baud_rate(dev->tty);
        *((uint32_t*)(outpacket+2))=cpu_to_le32((uint32_t)i);

    ON_COMMAND(buzy&LAVA_CMD_SETLCR);
        outpacketsize=9;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSSetLineControlComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        outpacket[1]=1;
        outpacket[1]=cpu_to_le16(outpacket[1]);
        //set data bits
        switch(dev->tty->termios->c_cflag&CSIZE){
            case CS8: ++outpacket[2];
            case CS7: ++outpacket[2];
            case CS6: ++outpacket[2];
        }
        //set stop bits
        if(dev->tty->termios->c_cflag&CSTOPB) outpacket[2]|=0x04;
        //set parity bits
        if(dev->tty->termios->c_cflag&PARENB) outpacket[2]|=0x08;
        if((dev->tty->termios->c_cflag&PARODD)==0) outpacket[2]|=0x10;
#ifdef CMSPAR
        if(dev->tty->termios->c_cflag&CMSPAR) outpacket[2]|=0x20;
#endif        
        outpacket[2]=cpu_to_le16(outpacket[2]);
        
    ON_COMMAND(buzy&LAVA_CMD_SETMCR);
        outpacketsize=9;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSSetModemControlComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        outpacket[1]=1;
        outpacket[1]=cpu_to_le16(outpacket[1]);
        //turn on RTS&DTR, enable interrupts
        if(dev->tty->termios->c_cflag&CBAUD) outpacket[2]=0x000b;
        else outpacket[2]=0x0008;
        outpacket[2]=cpu_to_le16(outpacket[2]);
    
    ON_COMMAND(buzy&LAVA_CMD_OPEN);
        //send open com port request
        outpacketsize=10;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSOpenComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        outpacket[1]=2;
        outpacket[1]=cpu_to_le16(outpacket[1]);
        outpacket[2]=1;
        outpacket[2]=cpu_to_le16(outpacket[2]);

    ON_COMMAND(buzy&LAVA_CMD_CLOSE);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSCloseComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);

    ON_COMMAND(buzy&LAVA_CMD_SUSPEND);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSSuspendShipRXComPortNRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        buzy&=~LAVA_CMD_SUSPEND;
        
    ON_COMMAND(buzy&LAVA_CMD_RESUME);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSResumeShipRXComPortNRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        buzy&=~LAVA_CMD_RESUME;
                
      ON_COMMAND(buzy&LAVA_CMD_ERROR);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSClearErrorComPortNRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        buzy&=~LAVA_CMD_ERROR;
        
    ON_COMMAND(buzy&LAVA_CMD_EVENT);
        outpacketsize=12;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSClrHistoryEventMaskComPortNRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        outpacket[2]=cRSGetModemStatusComPortCRequest;
        outpacket[2]=cpu_to_le16(outpacket[2]);
        
    ON_COMMAND(buzy&LAVA_CMD_SETBRK);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSTurnOnBreakComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);

    ON_COMMAND(buzy&LAVA_CMD_CLRBRK);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSTurnOffBreakComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);

    ON_COMMAND(buzy&LAVA_CMD_SETRTS);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSSetRtsComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);
        
    ON_COMMAND(buzy&LAVA_CMD_SETDTR);
        outpacketsize=8;
        memset(outpacket, 0, outpacketsize);
        outpacket[0]=cRSSetDtrComPortCRequest;
        outpacket[0]=cpu_to_le16(outpacket[0]);

    ON_COMMAND(!buzy && !txdisable);
        bptr=(uint8_t*)&outpacket[5];
        spin_lock(&dev->lock);
        n=CIRC_CNT(dev->xmit.head, dev->xmit.tail, PAGE_SIZE);
        if(n>0){
            //there is some data to send
            outpacket[0]=cRSCompleteWriteComPortNRequest;
            if(n>dev->dsize) {
                n=dev->dsize;
                outpacket[0]=cRSWriteComPortNRequest;
            }
            outpacket[0]=cpu_to_le16(outpacket[0]);
            i=CIRC_CNT_TO_END(dev->xmit.head, dev->xmit.tail, PAGE_SIZE);
            if(i>=n){
                //No wrap around
                memcpy(bptr, dev->xmit.buf+dev->xmit.tail, n);
            } else{
                memcpy(bptr, dev->xmit.buf+dev->xmit.tail, i);
                memcpy(bptr+i, dev->xmit.buf, n-i);
            }
            dev->xmit.tail=(dev->xmit.tail+n)&(PAGE_SIZE-1);
            dev->icount.tx+=n;
            spin_unlock(&dev->lock);
            for(i=0; i<4; i++) *(bptr+n+i)=0;
            outpacket[1]=n+6;
            outpacket[1]=cpu_to_le16(outpacket[1]);
            *((uint32_t*)&outpacket[2])=irp;
            outpacket[4]=n;
            outpacket[4]=cpu_to_le16(outpacket[4]);
            outpacketsize=n+14;    //10 bytes header and 4 bytes terminator
            txdisable=1;
            //notify tty layer
            if(CIRC_CNT(dev->xmit.head,dev->xmit.tail,PAGE_SIZE)<WAKEUP_CHARS){
                if((dev->tty->flags&(1<<TTY_DO_WRITE_WAKEUP))&&dev->tty->ldisc.write_wakeup)
                    (dev->tty->ldisc.write_wakeup)(dev->tty);
                wake_up_interruptible(&dev->tty->write_wait);
            }
        } else {
        //nothing to transmit
        //TODO: if not lava_sts_drained send "readlcr" request
            spin_unlock(&dev->lock);
        }

END_MESSAGE_MAP();

/*
 * Check if we have message (create sync packet if we dont) and send message
 */    
        if(!outpacketsize){
            if(TransportOffFlag){
                if(--watchdogtimer<0){
                    watchdogtimer=WATCHDOG;
                    outpacketsize=4;
                    memset(outpacket, 0, outpacketsize);                    
                } else continue;
            } else{
                if(emptytransaction){
                    outpacketsize=4;
                    memset(outpacket, 0, outpacketsize);
                } else{
                    outpacketsize=8;
                    memset(outpacket, 0, outpacketsize);
                    outpacket[0]=cRSTransportOffRequest;
                    outpacket[0]=cpu_to_le16(outpacket[0]);                
                }
            }
        }
#ifdef MDEBUG    
if(outpacketsize>4){    //filter out syncs
printk(KERN_INFO"%s: message to send (hex):", me->comm);
for (i=0; i<outpacketsize; i++) printk(" %02x", *((uint8_t*)outpacket + i));
printk("\n");
}
#endif
        TransportOffFlag=0;
        msg.msg_flags = MSG_DONTWAIT|MSG_NOSIGNAL;
        msg.msg_name = NULL;
        msg.msg_namelen  = 0;
        msg.msg_control = NULL;
        msg.msg_controllen = 0;
        msg.msg_iov     = &iov;
        msg.msg_iovlen = 1;
        msg.msg_iov->iov_base = (char*)outpacket;
        msg.msg_iov->iov_len = (__kernel_size_t)outpacketsize;
        if(dev->td->sk){
            oldfs = get_fs(); set_fs(KERNEL_DS);
            sock_sendmsg(dev->td,&msg,(size_t)outpacketsize);
            set_fs(oldfs);
        } else {
#ifdef MDEBUG
            printk(KERN_INFO"%s: TCPsend failed.\n", me->comm);
#endif
            CloseTunnel(dev->td);
            dev->td=NULL;
        }
        
    }// end of main loop
//exit
    if(dev->td) CloseTunnel(dev->td);    //close TCP socket
    lava_dev[dev->devno]=NULL;    //remove the device from table
    kfree(dev->xmit.buf);
    kfree(dev->tcpbuf);
    kfree(dev->usrbuf);
    kfree(dev);
    atomic_dec(&LTPCOUNT);
    wake_up(&lava_wait_threads);
    return 0;
}

/* *************************************************************************** */
static int
lava_open(struct tty_struct *tty, struct file * filp){

    ETHERSERIAL *dev;
    int status;
    
//    MOD_INC_USE_COUNT;
    if(!tty){
        printk("lava_open: tty* is NULL!\n");
        return (-EIO);
    }
    dev=lava_dev[PORTNO(tty)];
    if(!dev) return (-ENODEV);
    if(tty->count>1){
        if((dev->uid!=current->uid)&&
           (dev->uid!=current->euid)&&
           !capable(CAP_DAC_OVERRIDE)
        ) return -EBUSY;
        else return 0;
    }
    //first time open
    spin_lock(&dev->lock);
    dev->tty=tty;
    dev->uid=current->uid;
    dev->cmd=LAVA_CMD_OPEN;
    spin_unlock(&dev->lock);
    wake_up_interruptible(&dev->wait);
    sleep_on_timeout(&dev->open_wait, HZ*15);
    spin_lock(&dev->lock);
    status= (dev->status&LAVA_STS_OPEN)? 1:0;
    spin_unlock(&dev->lock);
    if(status){
        return 0;
    } else {
        spin_lock(&dev->lock);
        dev->cmd=0;
        spin_unlock(&dev->lock);
        return (-ENOLINK);    
    }
    
}

/* *************************************************************************** */
static void    
lava_close(struct tty_struct *tty, struct file * filp){

    ETHERSERIAL *dev;
    int status;

//    MOD_DEC_USE_COUNT;    
    if(!tty){
        printk("lava_close: tty* is NULL!\n");
        return;
    }
    dev=lava_dev[PORTNO(tty)];
    if(tty->count>1) return;
    //last time close
    tty->closing=1;
    spin_lock(&dev->lock);
    status= (dev->status&LAVA_STS_OPEN)? 1:0;
    spin_unlock(&dev->lock);
    if(!status){
        //likely we weren't able to open the port
        //(wrong IP, no ethernet connection, etc.)
        //printk("lava_close: trying to close closed port!\n");    
        return;
    }
    lava_wait_until_sent(tty, 0);
//    if(tty->driver.wait_until_sent) tty->driver.wait_until_sent(tty, 0);    
    spin_lock(&dev->lock);
    dev->cmd|=LAVA_CMD_CLOSE;
    spin_unlock(&dev->lock);
    wake_up_interruptible(&dev->wait);
    sleep_on_timeout(&dev->close_wait, HZ*25);
    spin_lock(&dev->lock);
    status= (dev->status&LAVA_STS_OPEN)? 1:0;
    //insane actions for sanity
    dev->status&=~LAVA_STS_OPEN;
    dev->xmit.head = dev->xmit.tail = 0;
    spin_unlock(&dev->lock);
    if(status) printk("lava_close: can't close port!\n");
    lava_flush_buffer(tty);
//    if (tty->driver.flush_buffer) tty->driver.flush_buffer(tty);
    if (tty->ldisc.flush_buffer) tty->ldisc.flush_buffer(tty);
    tty->closing=0;
    return;
}


/* *************************************************************************** */
static void    
lava_throttle(struct tty_struct *tty){

    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];

    if(dev){
        spin_lock(&dev->lock);
        dev->cmd|=LAVA_CMD_SUSPEND;
        spin_unlock(&dev->lock);
        wake_up_interruptible(&dev->wait);
    }
    return;
}

/* *************************************************************************** */
static void    
lava_unthrottle(struct tty_struct *tty){

    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];

    if(dev){
        spin_lock(&dev->lock);
        dev->cmd|=LAVA_CMD_RESUME;
        spin_unlock(&dev->lock);
        wake_up_interruptible(&dev->wait);
    }
    return;
}


/* *************************************************************************** */
static inline int
__lava_user_write(ETHERSERIAL *dev, const unsigned char *buf, int count){
    
    int c, ret = 0;
    
    if(down_interruptible(&dev->mux)) return -EINTR;
    while (1) {
        int c1;
        spin_lock(&dev->lock);
        c = CIRC_SPACE_TO_END(dev->xmit.head, dev->xmit.tail, PAGE_SIZE);
        spin_unlock(&dev->lock);
        if(count < c) c = count;
        if (c <= 0) break;
        c -= copy_from_user(dev->usrbuf, buf, c);
        if (!c) {
            if (!ret) ret = -EFAULT;
            break;
        }
        spin_lock(&dev->lock);
        c1 = CIRC_SPACE_TO_END(dev->xmit.head, dev->xmit.tail, PAGE_SIZE);
        if (c1 < c) c = c1;
        memcpy(dev->xmit.buf+dev->xmit.head, dev->usrbuf, c);
        dev->xmit.head=((dev->xmit.head+c)&(PAGE_SIZE-1));
        dev->status&=~LAVA_STS_DRAINED;
        spin_unlock(&dev->lock);
        buf += c;
        count -= c;
        ret += c;
    }
    up(&dev->mux);
    return ret;
}

/* *************************************************************************** */
static inline int
__lava_kern_write(ETHERSERIAL *dev, const unsigned char *buf, int count){
    
    int c, ret = 0;

    while (1) {
        spin_lock(&dev->lock);
        c = CIRC_SPACE_TO_END(dev->xmit.head, dev->xmit.tail, PAGE_SIZE);
        if (count < c) c = count;
        if (c <= 0){ 
            spin_unlock(&dev->lock);
            break;
        }
        memcpy(dev->xmit.buf+dev->xmit.head, buf, c);
        dev->xmit.head=((dev->xmit.head+c)&(PAGE_SIZE-1));
        dev->status&=~LAVA_STS_DRAINED;
        spin_unlock(&dev->lock);
        buf += c;
        count -= c;
        ret += c;
    }
    return ret;
}


/* *************************************************************************** */
static int
lava_write(struct tty_struct *tty, const unsigned char * buf, int count){
//lava_write(struct tty_struct *tty, int from_user, const unsigned char * buf, int count){
// removed "int from_user" for compatability with new kernel release
    
    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];
    int ret=-EIO;
    
    if(dev){
//        if (from_user)
//            ret = __lava_user_write(dev, buf, count);
//        else
        ret = __lava_kern_write(dev, buf, count);
        wake_up_interruptible(&dev->wait);
    }
    return ret;
}



/* *************************************************************************** */
static int
lava_write_room(struct tty_struct * tty){
    
    int c=0;
    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];
    
    if(dev){
        spin_lock(&dev->lock);
        c = CIRC_SPACE(dev->xmit.head, dev->xmit.tail, PAGE_SIZE);
        spin_unlock(&dev->lock);
    }
    return c;
}

/* *************************************************************************** */
static void
lava_flush_buffer(struct tty_struct * tty){

    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];

    if(dev){
        spin_lock(&dev->lock);
        dev->xmit.head = dev->xmit.tail = 0;
        spin_unlock(&dev->lock);
        wake_up_interruptible(&tty->write_wait);
        if((tty->flags&(1<<TTY_DO_WRITE_WAKEUP))&&tty->ldisc.write_wakeup) 
            (tty->ldisc.write_wakeup)(tty);
    }
    return;
}

/* *************************************************************************** */
static int
lava_chars_in_buffer(struct tty_struct * tty){

    int c=0;
    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];

    if(dev){
        spin_lock(&dev->lock);
        c = CIRC_CNT(dev->xmit.head, dev->xmit.tail, PAGE_SIZE);
        spin_unlock(&dev->lock);
    }
    return c;
}

/* *************************************************************************** */
static void
lava_flush_chars(struct tty_struct * tty){

    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];
    
    if(dev) wake_up_interruptible(&dev->wait);
//if not hw/sw flow stopped, kick tx;
//we don't actually need this
    return;
}

/* *************************************************************************** */
static int 
get_modem_info(ETHERSERIAL *dev, unsigned int *value){

    unsigned int result;
    uint16_t status;

    spin_lock(&dev->lock);
    status=dev->status;
    spin_unlock(&dev->lock);
    result =  ((status & LAVA_STS_RTS) ? TIOCM_RTS : 0)
        | ((status & LAVA_STS_DTR) ? TIOCM_DTR : 0)
        | ((status & LAVA_STS_DCD) ? TIOCM_CAR : 0)
        | ((status & LAVA_STS_RI)  ? TIOCM_RNG : 0)
        | ((status & LAVA_STS_DSR) ? TIOCM_DSR : 0)
        | ((status & LAVA_STS_CTS) ? TIOCM_CTS : 0);

    if (copy_to_user(value, &result, sizeof(int)))
        return -EFAULT;
    return 0;
}


/* *************************************************************************** */
static int 
set_modem_info(ETHERSERIAL *dev, unsigned int cmd, unsigned int *value){

    unsigned int arg;
    uint32_t command=0;

    if (copy_from_user(&arg, value, sizeof(int))) return -EFAULT;
    switch (cmd) {
    case TIOCMBIS: 
        if (arg & TIOCM_RTS)
            command |= LAVA_CMD_SETRTS;
        if (arg & TIOCM_DTR)
            command |= LAVA_CMD_SETDTR;
        break;
    case TIOCMBIC:
        if (arg & TIOCM_RTS)
            command |= LAVA_CMD_CLRRTS;
        if (arg & TIOCM_DTR)
            command |= LAVA_CMD_CLRDTR;
        break;
    case TIOCMSET:
        command =((arg & TIOCM_RTS) ? LAVA_CMD_SETRTS : LAVA_CMD_CLRRTS)
        |((arg & TIOCM_DTR) ? LAVA_CMD_SETDTR : LAVA_CMD_CLRDTR);
        break;
    default:
        return -EINVAL;
    }
    if(command){
        spin_lock(&dev->lock);
        dev->cmd|=command;
        spin_unlock(&dev->lock);
        wake_up_interruptible(&dev->wait);
    }
    return 0;
}

#define SERIAL_IO_ETH    4    //serial IO over ethernet
//SERIAL_IO_PORT 0

/* *************************************************************************** */
static int 
get_serial_info(ETHERSERIAL *dev, struct serial_struct *info){

    struct serial_struct tmp;
    
    if(!info) return -EFAULT;
    memset(&tmp, 0, sizeof(tmp));
    tmp.type           = PORT_16550A;
    tmp.line           = dev->devno;
    tmp.xmit_fifo_size = 16;
    tmp.baud_base      = 115200;
    tmp.close_delay    = 25*HZ;         //lava_close() max. timeout
    tmp.closing_wait   = 10*HZ;         //lava_wait_until_sent() max. timeout
    tmp.io_type        = SERIAL_IO_ETH;
    return (copy_to_user(info, &tmp, sizeof(*info)))? -EFAULT : 0;
}

/* *************************************************************************** */
/*
 * get_lsr_info - get line status register info
 *
 * Purpose: Let user call ioctl() to get info when the UART physically
 *         is emptied.  On bus types like RS485, the transmitter must
 *         release the bus after transmitting. This must be done when
 *         the transmit shift register is empty, not be done when the
 *         transmit holding register is empty.  This functionality
 *         allows an RS485 driver to be written in user space. 
 */
/* *************************************************************************** */
static int
get_lsr_info(ETHERSERIAL *dev, unsigned int *value){

    uint16_t status;
    int n;
    unsigned int result;

    spin_lock(&dev->lock);
    n=CIRC_CNT(dev->xmit.head, dev->xmit.tail, PAGE_SIZE);
    status=dev->status;
    spin_unlock(&dev->lock);    
    result= ((status&LAVA_STS_DRAINED)&&!n)? TIOCSER_TEMT : 0;
    if(copy_to_user(value, &result, sizeof(int))) return -EFAULT;
    return 0;
}

/* *************************************************************************** */
static int
lava_ioctl(struct tty_struct *tty, struct file * file,
            unsigned int cmd, unsigned long arg){
            
    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];
    struct serial_icounter_struct icount, cprev;
    
    if(!dev) return -EIO;
    if ((cmd != TIOCGSERIAL) && (cmd != TIOCSSERIAL) &&
        (cmd != TIOCSERCONFIG) && (cmd != TIOCSERGSTRUCT) &&
        (cmd != TIOCMIWAIT) && (cmd != TIOCGICOUNT)) {
        if (tty->flags & (1 << TTY_IO_ERROR))
            return -EIO;
    }
    
    switch (cmd) {
        case TIOCMGET:
            return get_modem_info(dev, (unsigned int *) arg);
        case TIOCMBIS:
        case TIOCMBIC:
        case TIOCMSET:
            return set_modem_info(dev, cmd, (unsigned int *) arg);

        /* 
         * Get counter of input serial line interrupts (DCD,RI,DSR,CTS)
         * Return: write counters to the user passed counter struct
         * NB: both 1->0 and 0->1 transitions are counted except for
         *     RI where only 0->1 is counted.
         */

        case TIOCGICOUNT:
            spin_lock(&dev->lock);
            icount=dev->icount;
            spin_unlock(&dev->lock);
            if (copy_to_user((void *)arg, &icount, sizeof(icount)))
                return -EFAULT;
            return 0;

        case TIOCGSERIAL:
            return get_serial_info(dev, (struct serial_struct *) arg);

        case TIOCSSERIAL:
        case TIOCSERGSTRUCT:
        case TIOCSERCONFIG:
            return -EPERM;

        case TIOCSERGETLSR: /* Get line status register */
            return get_lsr_info(dev, (unsigned int *) arg);
            
        case TIOCMIWAIT:
        /*
         * Wait for any of the 4 modem inputs (DCD,RI,DSR,CTS) to change
         * - mask passed in arg for lines of interest
          *   (use |'ed TIOCM_RNG/DSR/CD/CTS for masking)
         * Caller should use TIOCGICOUNT to see which one it was
         */
             spin_lock(&dev->lock);        
            /* note the counters on entry */
            cprev = dev->icount;
            spin_unlock(&dev->lock);
            while (1) {
                interruptible_sleep_on(&dev->delta_msr_wait);
                /* see if a signal did it */
                if (signal_pending(current)) return -ERESTARTSYS;
                spin_lock(&dev->lock);        
                icount = dev->icount; /* atomic copy */
                spin_unlock(&dev->lock);
                if (icount.rng == cprev.rng && icount.dsr == cprev.dsr && 
                    icount.dcd == cprev.dcd && icount.cts == cprev.cts)
                    return -EIO; /* no change => error */
                if ( ((arg & TIOCM_RNG) && (icount.rng != cprev.rng)) ||
                     ((arg & TIOCM_DSR) && (icount.dsr != cprev.dsr)) ||
                     ((arg & TIOCM_CD)  && (icount.dcd != cprev.dcd)) ||
                     ((arg & TIOCM_CTS) && (icount.cts != cprev.cts)) ) {
                    return 0;
                }
                cprev = icount;
            }
            /* NOTREACHED */        

        default:
            return -ENOIOCTLCMD;
        }
    return 0;
}

/* *************************************************************************** */
static void
lava_set_termios(struct tty_struct *tty, struct ktermios *old_termios){
//lava_set_termios(struct tty_struct *tty, struct termios *old_termios){
//  changed to "struct ktermios *old_termios" for compatability with new kernel
    
    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];
        
    if(dev){
        //seems linux apps rely on this smart behaviour
	if(old_termios){
            if((tty->termios->c_cflag==old_termios->c_cflag)&&
            (RELEVANT_IFLAG(tty->termios->c_iflag)==
             RELEVANT_IFLAG(old_termios->c_iflag))
            ) return;    
	}
        spin_lock(&dev->lock);
        dev->cmd|=LAVA_CMD_INIT;
        spin_unlock(&dev->lock);
        wake_up_interruptible(&dev->wait);
    }
    return;
}

/* *************************************************************************** */
static void
lava_stop(struct tty_struct *tty){
    return;
}

/* *************************************************************************** */
static void    
lava_start(struct tty_struct *tty){
    return;
}
    
/* *************************************************************************** */
static void
lava_hangup(struct tty_struct *tty){

    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];
    
    if(dev){
        lava_flush_buffer(tty);
        wake_up_interruptible(&dev->delta_msr_wait);
    }
    return;
}

/* *************************************************************************** */
static void
lava_break(struct tty_struct *tty, int break_state){

    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];

    if(dev){
        spin_lock(&dev->lock);
        dev->cmd|=(break_state==-1)?LAVA_CMD_SETBRK:LAVA_CMD_CLRBRK;
        spin_unlock(&dev->lock);
        wake_up_interruptible(&dev->wait);
    }
    return;
}

/* *************************************************************************** */
static void    
lava_send_xchar(struct tty_struct *tty, char ch){
    return;
//we do not handle software flow control,
//remote server does.
}

/* *************************************************************************** */
static void    
lava_wait_until_sent(struct tty_struct *tty, int timeout){

    ETHERSERIAL *dev=lava_dev[PORTNO(tty)];
    unsigned long orig_jif, checkperiod;
    uint16_t status;
    
    //stty utility calls this function with 
    //insane parameter timeout=2147483647
    if(!timeout || timeout>10*HZ) timeout=10*HZ;
    checkperiod=HZ/50;
    if(timeout<checkperiod) checkperiod=timeout;
    orig_jif=jiffies;
    do{
        if (signal_pending(current)) break;    
        spin_lock(&dev->lock);
        status=dev->status;
        spin_unlock(&dev->lock);
        if(status&LAVA_STS_DRAINED) break;
        set_current_state(TASK_INTERRUPTIBLE);
        schedule_timeout(checkperiod);
        if (time_after(jiffies, orig_jif + timeout)) break;
    } while(1);
    return;
}

/* *************************************************************************** */
static inline int
line_info(char *buf, ETHERSERIAL *dev){

    int        ret;
    char    stat_buf[30];
    uint16_t status;
    
    if(!dev) return 0;
    ret=sprintf(buf, "%s%d", name, dev->devno);
    ke_inet_name(stat_buf, dev->ip);
    ret+=sprintf(buf+ret, " %s:%d", stat_buf, dev->port);
    spin_lock(&dev->lock);
    status=dev->status;
    spin_unlock(&dev->lock);
    
    ret+=sprintf(buf+ret, " %s", (status&LAVA_STS_OPEN) ? "open":"closed");
    
    stat_buf[0] = 0;
    stat_buf[1] = 0;
    if (status & LAVA_STS_RTS)
        strcat(stat_buf, "|RTS");
    if (status & LAVA_STS_CTS)
        strcat(stat_buf, "|CTS");
    if (status & LAVA_STS_DTR)
        strcat(stat_buf, "|DTR");
    if (status & LAVA_STS_DSR)
        strcat(stat_buf, "|DSR");
    if (status & LAVA_STS_DCD)
        strcat(stat_buf, "|CD");
    if (status & LAVA_STS_RI)
        strcat(stat_buf, "|RI");


    ret += sprintf(buf+ret, " tx:%d rx:%d",
        dev->icount.tx, dev->icount.rx);

    ret+=sprintf(buf+ret, " dcts:%d ddcd:%d ddsr:%d teri:%d",
        dev->icount.cts, dev->icount.dcd, dev->icount.dsr, dev->icount.rng);
    
     
    if (dev->icount.frame)
        ret += sprintf(buf+ret, " fe:%d", dev->icount.frame);
    
    if (dev->icount.parity)
        ret += sprintf(buf+ret, " pe:%d", dev->icount.parity);
    
    if (dev->icount.brk)
        ret += sprintf(buf+ret, " brk:%d", dev->icount.brk);    

    if (dev->icount.overrun)
        ret += sprintf(buf+ret, " oe:%d", dev->icount.overrun);

    ret += sprintf(buf+ret, " %s\n", stat_buf+1);
    return ret;
}

/* *************************************************************************** */
static int
lava_read_proc(char *page, char **start, off_t off, int count,
            int *eof, void *data){

    int i, len = 0, l;
    off_t    begin = 0;

    len += sprintf(page, 
    LAVADRIVERNAME " $Revision: " LAVAVERSION "$ " LAVADATE "\n");
    for (i = 0; i < NR_PORTS && len < 4000; i++) {
        l = line_info(page + len, lava_dev[i]);
        len += l;
        if (len+begin > off+count)
            goto done;
        if (len+begin < off) {
            begin += len;
            len = 0;
        }
    }
    *eof = 1;
done:
    if (off >= len+begin) return 0;
    *start = page + (off-begin);
    return ((count < begin+len-off) ? count : begin+len-off);
}
            
/* *************************************************************************** */
static int __init 
lava_init(void){

    char *optarg, *optnext;
    int i=0;
    int ret=0;
    ETHERSERIAL *dev;
        
    atomic_set(&lava_shutdown_cmd, 0);
    atomic_set(&LTPCOUNT, 0);
            
    printk(KERN_INFO LAVADRIVERNAME " $Revision: " LAVAVERSION "$ " LAVADATE "\n");

    //parse ip string
    while((optarg=ip[i]) && i<NR_PORTS){
        dev=(ETHERSERIAL*)kmalloc(sizeof(ETHERSERIAL), GFP_KERNEL);
        if(!dev){
            ret=-ENOMEM;
            break;
        }
        memset(dev, 0, sizeof(ETHERSERIAL));
        dev->xmit.buf=(char*)kmalloc(PAGE_SIZE, GFP_KERNEL);
        if(!dev->xmit.buf){
            kfree(dev);
            ret=-ENOMEM;    
            break;
        }
        dev->tcpbuf=(uint8_t*)kmalloc(PAGE_SIZE, GFP_KERNEL);
        if(!dev->tcpbuf){
            kfree(dev->xmit.buf);
            kfree(dev);
            ret=-ENOMEM;
            break;
        }
        dev->usrbuf=(uint8_t*)kmalloc(PAGE_SIZE, GFP_KERNEL);
        if(!dev->usrbuf){
            kfree(dev->xmit.buf);
            kfree(dev->tcpbuf);
            kfree(dev);
            ret=-ENOMEM;
            break;
        }
        optnext=strchr(optarg, ':');
        if(optnext){
            *optnext=0;
            dev->port=simple_strtoul(optnext+1,NULL,0);
        } else dev->port=INITPORT;
        dev->ip=ke_inet_addr(optarg);
        if(dev->ip==0||dev->ip==0xFFFFFFFF){
            kfree(dev->xmit.buf);
            kfree(dev->tcpbuf);
            kfree(dev->usrbuf);
            kfree(dev);
            ret=-EINVAL;
            break;
        }
        init_waitqueue_head(&dev->wait);
        init_waitqueue_head(&dev->open_wait);
        init_waitqueue_head(&dev->close_wait);
        init_waitqueue_head(&dev->delta_msr_wait);
        init_MUTEX(&dev->mux);
        spin_lock_init(&dev->lock);
        dev->devno=i;
        if(kernel_thread(ltp, (void*) dev, CLONE_KERNEL)<0){
            kfree(dev->xmit.buf);
            kfree(dev->tcpbuf);
            kfree(dev->usrbuf);
            kfree(dev);
            ret=-ECHILD;
            break;
        }
        if(verbose){
            printk(KERN_INFO "ltpd-%02d started: device %s:%d (/dev/%s%d).\n", 
                i, optarg, dev->port, name, i);
        }
        lava_dev[i]=dev;
        atomic_inc(&LTPCOUNT);
        i++;
    }
        
    if(!atomic_read(&LTPCOUNT)){
        //none of the threads launched
        return ret?ret:-EINVAL;
    }

    //register tty driver
    if(!ret){
        memset(&lava_driver, 0, sizeof(struct tty_driver));

        lava_driver.magic                = TTY_DRIVER_MAGIC;       /* magic number for this structure */
//        lava_driver.cdev                 = cdev;  
//        lava driver.owner                = owner;
        lava_driver.driver_name          = driver_name;
        lava_driver.name                 = name;
//        lava_driver.name_base            = name_base;              /* offset of printed name */
        lava_driver.major                = ttymajor;               /* major device number */
        lava_driver.minor_start          = 0;                      /* start of minor device number */
//        lava_driver.minor_num            = minor_num;              /* number of possible devices */
        lava_driver.minor_num            = 4;                      /* number of possible devices */
        lava_driver.num                  = atomic_read(&LTPCOUNT); /* number of devices allocated */
        lava_driver.type                 = TTY_DRIVER_TYPE_SERIAL; /* type of tty driver */
        lava_driver.subtype              = SERIAL_TYPE_NORMAL;     /* subtype of tty driver */
// struct ktermios init_termios                                    /*  Initial termios settings. */
        lava_driver.init_termios         = tty_std_termios;
        lava_driver.init_termios.c_iflag = 0;
        lava_driver.init_termios.c_oflag = 0;
        lava_driver.init_termios.c_cflag = B9600 | CS8 | CREAD | CLOCAL | HUPCL;
        lava_driver.init_termios.c_lflag = 0;
                                                                   /* tty driver flags */
        lava_driver.flags                = TTY_DRIVER_REAL_RAW | TTY_DRIVER_DYNAMIC_DEV;
//        lava_driver.refcount             = &lava_refcount;         /* for loadable tty drivers */
// struct proc_dir_entry *proc_entry                                 /* /proc fs entry goes here */
// struct tty_driver *othera                                         /* only used for pty driver */

        lava_driver.ttys                 = lava_table;
        lava_driver.termios              = lava_termios;
        lava_driver.termios_locked       = lava_termios_locked;
//        lava_driver.driver_state         = driver_state            /* only used for the pty driver */

	tty_set_operations(&lava_driver, &lava_ops);

/*
*       lava_driver.open            = lava_open;
*       lava_driver.close           = lava_close;
*       lava_driver.throttle        = lava_throttle;
*       lava_driver.unthrottle      = lava_unthrottle;
*       lava_driver.write           = lava_write;
*       lava_driver.write_room      = lava_write_room;
*       lava_driver.flush_buffer    = lava_flush_buffer;
*       lava_driver.chars_in_buffer = lava_chars_in_buffer;
*       lava_driver.flush_chars     = lava_flush_chars;
*       lava_driver.ioctl           = lava_ioctl;
*       lava_driver.set_termios     = lava_set_termios;
*       lava_driver.stop            = lava_stop;
*       lava_driver.start           = lava_start;
*       lava_driver.hangup          = lava_hangup;
*       lava_driver.break_ctl       = lava_break;
*       lava_driver.send_xchar      = lava_send_xchar;
*       lava_driver.wait_until_sent = lava_wait_until_sent;
*       lava_driver.read_proc       = lava_read_proc;
*/
        if((ret=tty_register_driver(&lava_driver)))
            printk(KERN_ERR "Couldn't register" LAVADRIVERNAME "!\n");
    }
    
    if(ret){
        //some threads have been launched,
        //but we have some error
        printk(LAVADRIVERNAME ": failed to start.\n");
        atomic_inc(&lava_shutdown_cmd);
        for(i=0; i<NR_PORTS; i++)
            if(lava_dev[i]) wake_up_interruptible(&lava_dev[i]->wait);    
        wait_event(lava_wait_threads, !atomic_read(&LTPCOUNT));
    } else {
        if(verbose) printk(KERN_INFO "Lava Etherserial TTY devices major number is %d.\n", ttymajor);
    }
    return ret;
}

/* *************************************************************************** */
static void __exit 
lava_cleanup(void){

    int i;
    
    atomic_inc(&lava_shutdown_cmd);
    for(i=0; i<NR_PORTS; i++)
        if(lava_dev[i]) wake_up_interruptible(&lava_dev[i]->wait);
    wait_event(lava_wait_threads, !atomic_read(&LTPCOUNT));
    printk(KERN_INFO LAVADRIVERNAME " removed.\n");
    tty_unregister_driver(&lava_driver);
}

module_init(lava_init);
module_exit(lava_cleanup);

/* *************************************************************************** */
